<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       http://www.castorstudio.com
 * @since      1.0.0
 *
 * @package    Uwac
 * @subpackage Uwac/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Uwac
 * @subpackage Uwac/includes
 * @author     CastorStudio <support@castorstudio.com>
 */
class Uwac {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      Uwac_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;


	/**
	 * Themes used on the plugin
	 */
	private static $themes;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		if ( defined( 'UWAC_VERSION' ) ) {
			$this->version = UWAC_VERSION;
		} else {
			$this->version = '1.0.0';
		}
		$this->plugin_name = 'uwac';

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_public_hooks();

	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Uwac_Loader. Orchestrates the hooks of the plugin.
	 * - Uwac_i18n. Defines internationalization functionality.
	 * - Uwac_Admin. Defines all hooks for the admin area.
	 * - Uwac_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-uwac-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-uwac-i18n.php';



		/**
		 * El archivo responsable de cargar el framework de Admin y los archivos externos necesarios
		 * para hacer funcionar el plugin.
		 * 
		 * Se agrega aquí para poder tener disponibles las funciones antes de llamar a las acciones
		 * del área de administración y del área pública
		 * 
		 * @date 22/6/2018
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/includes/dependencies.php';



		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-uwac-admin.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-uwac-public.php';

		/**
		 * Common Classes
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'common/class-uwac-common.php';


		/**
		 * UWAC THEMES
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-uwac-themes.php';

		$this->loader = new Uwac_Loader();

		$this::$themes = new Uwac_Themes();

	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the Uwac_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function set_locale() {

		$plugin_i18n = new Uwac_i18n();

		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );

	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {

		$plugin_admin = new Uwac_Admin( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );

		// Framework Hooks
		$this->loader->add_action('cssf_framework_load_config', $plugin_admin, 'load_plugin_config');

		// Admin Menu Pages
		$this->loader->add_action( 'admin_menu', $plugin_admin, 'register_admin_pages',990);

		if (cs_uwac_get_settings('plugin_status') == 'enabled'){
			/**
			 * Restrict Settings Access
			 * 
			 * @since 1.1.0
			 */ 
			$this->loader->add_filter('cssframework/cs-uwac/show_admin',$plugin_admin,'show_admin_menu',2);
	
			
			/**
			 * Admin Widget View
			 */
			// $this->loader->add_action('in_admin_footer', $plugin_admin, 'floating_widget_view',1);
			$this->loader->add_action('in_admin_header', $plugin_admin, 'floating_widget_view');
	
	
			/**
			 * Plugin Info 
			 * Filters the plugin action links on "Plugins" page
			 * 
			 * 1. Filter for plugin action links 	- Hook: plugin_action_links
			 * 2. Filter for plugin meta links 		- Hook: plugin_row_meta
			 * 
			 * @since 1.0.0
			 */
			$this->loader->add_filter( 'plugin_action_links', $plugin_admin, 'plugin_row_action_links', 10, 2 );
			$this->loader->add_filter( 'plugin_row_meta', $plugin_admin, 'plugin_row_meta_links' , 10, 2 );
		}
	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_public_hooks() {

		$plugin_public = new Uwac_Public( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );

		if (cs_uwac_get_settings('plugin_status') == 'enabled'){
			// Widget View
			$this->loader->add_action('wp_footer',$plugin_public,'floating_widget_view');
	
			/**
			 * AJAX CALLS
			 * 
			 * 1. Dynamic Themes Stylesheets
			 * 2. Dynamic Public Themes Stylesheets
			 * 
			 * @since 1.0.0
			 */
			$this->loader->add_action('wp_ajax_uwac_dynamic_themes',$plugin_public,'dynamic_themes_callback');
			$this->loader->add_action('wp_ajax_nopriv_uwac_dynamic_themes',$plugin_public,'dynamic_themes_callback');
	
	
			// Shortcodes
			$this->loader->add_shortcode('uwac_agent', $plugin_public, 'shortcode_agent');
			$this->loader->add_shortcode('uwac_widget', $plugin_public, 'shortcode_widget');
	
			// WooCommerce
			$this->loader->add_action( 'woocommerce_single_product_summary', $plugin_public, 'before_single_product_summary', 20);
			$this->loader->add_action( 'woocommerce_single_product_summary', $plugin_public, 'after_single_product_summary', 25);
			$this->loader->add_action( 'woocommerce_before_add_to_cart_form', $plugin_public, 'before_atc_form', 5 );
			$this->loader->add_action( 'woocommerce_before_add_to_cart_button', $plugin_public, 'before_atc_button', 5 );
			$this->loader->add_action( 'woocommerce_after_add_to_cart_form', $plugin_public, 'after_atc_form', 5 );
			$this->loader->add_action( 'woocommerce_after_add_to_cart_button', $plugin_public, 'after_atc_button', 5 );
			$this->loader->add_action( 'woocommerce_product_meta_start', $plugin_public, 'product_meta_start', 5 );
			$this->loader->add_action( 'woocommerce_product_meta_end', $plugin_public, 'product_meta_end', 5 );
			$this->loader->add_action( 'woocommerce_after_single_product_summary', $plugin_public, 'woocommerce_before_tabs', 10 );
			$this->loader->add_action( 'woocommerce_after_single_product_summary', $plugin_public, 'woocommerce_after_tabs', 15 );
			$this->loader->add_action( 'woocommerce_after_single_product_summary', $plugin_public, 'woocommerce_before_related_products', 19 );
			$this->loader->add_action( 'woocommerce_after_single_product_summary', $plugin_public, 'woocommerce_after_related_products', 25 );
		}
	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    Uwac_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}


	/**
	 * Retrieve the specific themes instance of the plugin.
	 *
	 * @since     1.0.0
	 * @return    Uwac_Module
	 */
	static function get_themes(){
		return self::$themes;
	}
	static function get_theme($theme){
		return self::$themes->$theme;
	}

}
